#oodist: *** DO NOT USE THIS VERSION FOR PRODUCTION ***
#oodist: This file contains OODoc-style documentation which will get stripped
#oodist: during its release in the distribution.  You can use this file for
#oodist: testing, however the code of this development version may be broken!

package OODoc::Format;
use parent 'OODoc::Object';

use strict;
use warnings;

use Log::Report    'oodoc';

use OODoc::Manifest ();
use Scalar::Util    qw/weaken/;

our %formatters = (
	pod   => 'OODoc::Format::Pod',
	pod2  => 'OODoc::Format::Pod2',
	pod3  => 'OODoc::Format::Pod3',
	html  => 'OODoc::Format::Html',
	html2 => 'OODoc::Format::Html2',   # not (yet) included in the OODoc release
);

#--------------------
=chapter NAME

OODoc::Format - base class for all OODoc formatters

=chapter SYNOPSIS

  # Never instantiated directly.

=chapter DESCRIPTION

A formater produces manual pages in some way or an other which contain
(parts of) the module documentation.  Each formatter class is based on
this OODoc::Format class, which should not be instantiated directly.
By far most users will never explicitly create a formatter by themselves:
it is created implicitly when M<OODoc::formatter()> is called on a M<OODoc>
object.

Currently available formatters:

=over 4

=item * OODoc::Format::Pod

Simple formatter, which has the layout of the produced POD manual pages
hard-coded in it's body.  The only way to adapt the structure of the
pages is by extending the class, and thereby overrule some of the
methods which produce the text.  Not much of a problem for experienced
Object Oriented programmers.

=item * OODoc::Format::Pod2

This formatter uses the same methods to generate the manual page as
defined by OODoc::Format::Pod, but the general layout of the page
can be configured using templates.

You have to install L<Bundle::Template::Magic> to use this feature.

=item * OODoc::Format::Pod3

The whole formatter, implemented as template in OODoc::Template, a
very light weighted template system.

You have to install L<OODoc::Template> to use this feature.

=item * OODoc::Format::Html

Produce HTML by filling in templates. This module requires
L<Bundle::Template::Magic> and the ability to run cgi scripts.

=back

=chapter METHODS

=section Constructors

=c_method new %options

=requires index OODoc::Index
[2.01] The knowledge of the collected documentation.

=requires format 'pod*'|'html*'|PACKAGE
Select the formatter to be used.

=requires project STRING
The short name of this project (module), set by M<OODoc::new(project)>.

=requires version STRING
Many manual pages will contain the version of the project.  This can
be any STRING, although blanks are not advised.

=requires workdir DIRECTORY
The DIRECTORY where the output will be placed.  If it does not exist,
it will be created for you.

=option  manifest OBJECT
=default manifest undef

=error no formatter specified
=error no index specified

=error formatter has no project name
A formatter was created without a name specified for the project at
hand.  This should be passed with M<new(project)>.

=error no working directory specified for $name
The formatter has to know where the output can be written.  This
directory must be provided via M<new(workdir)>, but was not specified.

=error formatter for $name does not know the version

=error formatter $name has compilation errors: $err
The formatter which is specified does not compile, so can not be used.
=cut

sub new($%)
{	my ($class, %args) = @_;

	$class eq __PACKAGE__
		or return $class->SUPER::new(%args);

	my $format = $args{format} or error __x"no formatter specified";
	my $pkg = $formatters{$format} || $format;

	eval "require $pkg";
	$@ and error __x"formatter {name} has compilation errors: {err}", name => $format, err => $@;

	$pkg->new(%args);
}

sub init($)
{	my ($self, $args) = @_;

	$self->SUPER::init($args) or return;
	$self->{OF_format}   = delete $args->{format};

	my $name = $self->{OF_project} = delete $args->{project}
		or error __x"formatter knows no project name";

	$self->{OF_version}  = delete $args->{version}
		or error __x"formatter for {name} does not know the version", name => $name;

	$self->{OF_workdir}  = delete $args->{workdir}
		or error __x"no working directory specified for {name}", name => $name;

	$self->{OF_manifest} = delete $args->{manifest} || OODoc::Manifest->new;

	$self->{OF_index}    = delete $args->{index}  or error __x"no index specified";
	weaken($self->{OF_index});

	$self;
}

sub publish { panic }

#--------------------
=section Attributes

=method project
Returns the name of this project.
=cut

sub project() { $_[0]->{OF_project} }

=method version
Returns the version string of this project.  This version may
contains any character, so should be handled with care.

=method workdir
Returns the name of the work directory: the top location for all
the output files.

=method manifest
Returns the OODoc::Manifest object which maintains the names
of created files.

=method format
Code for the format.

=method index
[2.01] The collected documentation.
=cut

sub version()  { $_[0]->{OF_version} }
sub workdir()  { $_[0]->{OF_workdir} }
sub manifest() { $_[0]->{OF_manifest} }
sub format()   { $_[0]->{OF_format} }
sub index()    { $_[0]->{OF_index} }

#--------------------
=section Page generation

=method createPages %options

=option  append STRING|CODE
=default append undef
The value is passed on to M<createManual(append)>,
but the behavior is formatter dependent.

=option  manual_templates DIRECTORY
=default manual_templates undef
Passed to M<createManual(template)>, and defines the
location of the set of pages which has to be created for each manual
page.  Some formatters do not support templates and the valid values
are formatter dependent.

=option  other_templates DIRECTORY
=default other_templates undef
Other files which have to be copied
passed to M<createOtherPages(source)>.

=option  process_files REGEXP
=default process_files <formatter dependent>
Selects the files which are to be processed for special markup information.
Other files, like image files, will be simply copied.  The value will be
passed to M<createOtherPages(process)>.

=option  manual_format ARRAY
=default manual_format []
Options passed to M<createManual()> when a manual page has to be produced.
See the applicable formatter manual page for the possible flags and
values.

=option  select CODE|REGEXP
=default select <all>
Produce only the indicated manuals, which is useful in case of merging
manuals from different distributions.  When a REGEXP is provided, it
will be checked against the manual name.  The CODE reference will be
called with a manual as only argument.

=cut

sub createPages(%)
{	my ($self, %args) = @_;

	my $sel = $args{select} || sub { 1 };
	my $select = ref $sel eq 'CODE' ? $sel : sub { $_[0]->name =~ $sel };

	# Manual knowledge is global

	my $options = $args{manual_format} || [];
	my $index   = $self->index;

	foreach my $package (sort $index->packageNames)
	{
		foreach my $manual ($index->manualsForPackage($package))
		{	$select->($manual) or next;

			unless($manual->chapters)
			{	trace "  skipping $manual: no chapters";
				next;
			}

			trace "  creating manual $manual with ".(ref $self);

			$self->createManual(
				manual   => $manual,
				template => $args{manual_templates},
				append   => $args{append},
				@$options
			);
		}
	}

	#
	# Create other pages
	#

	trace "creating other pages";
	$self->createOtherPages(source => $args{other_templates}, process => $args{process_files});

	1;
}

=method createManual %options
Format a manual into the selected format.

=requires manual MANUAL
The manual to be formatted.

=option  append STRING|CODE
=default append undef

=option  template LOCATION
=default template undef
Some formatters support templates to descibe the output of the pages.
The valid values for this option differs per formatter.

=requires project STRING
The name of this project, which will appear on many pages.

=cut

sub createManual(@) {panic}

=method cleanup $manual, $text, %options
Takes the $text and cleans it up to be in the right format for the current
output syntax.  The cleaning up is parser dependent, and therefore the
parser of the manual is addressed to do the work.

=requires create_link CODE
This CODE reference is used to create an absolute link to a text
fragment.  It is called with the manual and the object where is
linked to (any OODoc::Text object).  As third parameter,
a string can be provided which should be displayed instead of the
object name.  Finally, all %options are passed as settings, by
reference to a HASH.

=option  tag $word|\@words
=default tag undef
Tag the produced block with this word, when possible.  The HTML renderer
will use this to add C<< class="@tags" >> to the resulting block, for
instance.
=cut

sub cleanup($$%) { ... }

=method cleanupString $manual, $string, %options
Format a single line piece of text.  The %options are passed to M<cleanup()>.
=cut

sub cleanupString($$%) { shift->cleanup(@_) }

=method showChapter %options
You can pass all %options about formatting to this method.  They will passed
to the related methods.  So: the list of options you can pass here is much
longer: the combination of everything possible for all show* methods.

In the %options, C<$BLOCK> can be C<chapter>, C<section>, C<subsection>,
and C<subsubsection>.

=requires chapter CHAPTER
=requires output FILE
=requires manual MANUAL

=option   show_inherited  'NO'|'REFER'|'EXPAND'
=default  show_inherited 'REFER'
REFER means that inherited sections are simply listed as reference
to the manual page which describes it.

=option   show_inherited_$BLOCK 'NO'|'REFER'|'EXPAND'
=default  show_inherited_$BLOCK <show_inherited>

=option   show_examples 'NO'|'EXPAND'
=default  show_examples 'EXPAND'

=option   show_examples_$BLOCK 'NO'|'REFER'|'EXPAND'
=default  show_examples_$BLOCK <show_examples>
=cut

sub showChapter(@)
{	my ($self, %args) = @_;
	my $chapter  = $args{chapter} or panic;
	my $manual   = $args{manual}  or panic;

	my $show_inh = $args{show_inherited};
	my $show_ch  = $args{show_inherited_chapter}    || $show_inh;
	my $show_sec = $args{show_inherited_section}    || $show_inh;
	my $show_ssec  = $args{show_inherited_subsection}    || $show_inh;
	my $show_sssec = $args{show_inherited_subsubsection} || $show_inh;

	my $show_examples = $args{show_examples} || 'EXPAND';

	if($manual->inherited($chapter))
	{	return $self if $show_ch eq 'NO';
		$self->showStructureRefer(%args, structure => $chapter);
		return $self;
	}

	$self->showStructureExpanded(%args, structure => $chapter,
		show_examples => $args{show_chapter_examples} || $show_examples,
	);

	foreach my $section ($chapter->sections)
	{	if($manual->inherited($section))
		{	next if $show_sec eq 'NO';
			if($show_sec ne 'REFER')
			{	$self->showStructureRefer(%args, structure => $section);
				next;
			}
		}

		$self->showStructureExpanded(%args, structure => $section,
			show_examples => $args{show_section_examples} || $show_examples,
		);

		foreach my $subsection ($section->subsections)
		{	if($manual->inherited($subsection))
			{	next if $show_ssec eq 'NO';
				if($show_ssec ne 'REFER')
				{	$self->showStructureRefer(%args, structure => $subsection);
					next;
				}
			}

			$self->showStructureExpanded(%args, structure => $subsection,
				show_examples => $args{show_subsection_examples} || $show_examples,
			);

			foreach my $subsubsection ($subsection->subsubsections)
			{	if($manual->inherited($subsubsection))
				{	next if $show_sssec eq 'NO';
					if($show_sssec ne 'REFER')
					{	$self->showStructureRefer(%args, structure => $subsubsection);
						next;
					}
				}

				$self->showStructureExpanded(%args, structure => $subsubsection,
					show_examples => $args{show_subsubsection_examples} || $show_examples,
				);
			}
		}
	}
}

=method showStructureExpanded %options
=option   show_examples 'NO'|'EXPAND'
=default  show_examples 'EXPAND'
=cut

sub showStructureExpanded(@) {panic}

=method showStructureRefer %options
=cut

sub showStructureRefer(@) {panic}

sub chapterName(@)        { $_[0]->showRequiredChapter(NAME        => @_) }
sub chapterSynopsis(@)    { $_[0]->showOptionalChapter(SYNOPSIS    => @_) }
sub chapterInheritance(@) { $_[0]->showOptionalChapter(INHERITANCE => @_) }
sub chapterDescription(@) { $_[0]->showRequiredChapter(DESCRIPTION => @_) }
sub chapterOverloaded(@)  { $_[0]->showOptionalChapter(OVERLOADED  => @_) }
sub chapterMethods(@)     { $_[0]->showOptionalChapter(METHODS     => @_) }
sub chapterExports(@)     { $_[0]->showOptionalChapter(EXPORTS     => @_) }
sub chapterDiagnostics(@) { $_[0]->showOptionalChapter(DIAGNOSTICS => @_) }
sub chapterDetails(@)     { $_[0]->showOptionalChapter(DETAILS     => @_) }
sub chapterReferences(@)  { $_[0]->showOptionalChapter(REFERENCES  => @_) }
sub chapterCopyrights(@)  { $_[0]->showOptionalChapter(COPYRIGHTS  => @_) }

=method showRequiredChapter $name, %options
=error missing required chapter $name in $manual
=cut

sub showRequiredChapter($%)
{	my ($self, $name, %args) = @_;
	my $manual  = $args{manual} or panic;

	my $chapter = $manual->chapter($name)
		or (error __x"missing required chapter {name} in {manual}", name => $name, manual => $manual), return;

	$self->showChapter(chapter => $chapter, %args);
}

=method showOptionalChapter $name, %options
=cut

sub showOptionalChapter($@)
{	my ($self, $name, %args) = @_;
	my $manual  = $args{manual} or panic;
	my $chapter = $manual->chapter($name) or return;
	$self->showChapter(chapter => $chapter, %args);
}

=method createOtherPages %options
Create other pages which come with the set of formatted manuals.  What
the contents of these pages is depends on the formatter.  Some formatters
simply ignore the functionality of this method as a whole: they do not
support data-files which are not manuals.

=option  source   DIRECTORY
=default source   undef
The location of the DIRECTORY which contains files which are part of
the produced set of documentation, but not copied per manual page
but only once.

=option  process  REGEXP
=default process  undef
Selects files to be processed from the source directory.  Other files
are copied without modification.  What happens with the selected
files is formatter dependent.

=cut

sub createOtherPages(@) { $_[0] }

=method showSubroutines %options

=option  subroutines ARRAY
=default subroutines []

=option  output  FILE
=default output  <selected filehandle>

=requires manual  MANUAL

=option  show_subs_index 'NO'|'NAMES'|'USE'
=default show_subs_index 'NO'

=option  show_inherited_subs 'NO'|'NAMES'|'USE'|'EXPAND'
=default show_inherited_subs 'USE'

=option  show_described_subs 'NO'|'NAMES'|'USE'|'EXPAND'
=default show_described_subs 'EXPAND'

=option  show_option_table 'NO'|'DESCRIBED'|'INHERITED'|'ALL'
=default show_option_table 'ALL'

=option  show_inherited_options 'NO'|'LIST'|'USE'|'EXPAND'
=default show_inherited_options 'USE'

=option  show_described_options 'NO'|'LIST'|'USE'|'EXPAND'
=default show_described_options 'EXPAND'

=option  show_examples 'NO'|'EXPAND'
=default show_examples 'EXPAND'

=option  show_diagnostics 'NO'|'EXPAND'
=default show_diagnostics 'NO'

=cut

sub showSubroutines(@)
{	my ($self, %args) = @_;

	my @subs   = $args{subroutines} ? sort @{$args{subroutines}} : [];
	@subs or return $self;

	my $manual = $args{manual} or panic;
	my $output = $args{output}    || select;

	# list is also in ::Pod3
	$args{show_described_options} ||= 'EXPAND';
	$args{show_described_subs}    ||= 'EXPAND';
	$args{show_diagnostics}       ||= 'NO';
	$args{show_examples}          ||= 'EXPAND';
	$args{show_inherited_options} ||= 'USE';
	$args{show_inherited_subs}    ||= 'USE';
	$args{show_option_table}      ||= 'ALL';
	$args{show_subs_index}        ||= 'NO';

	$self->showSubsIndex(%args, subroutines => \@subs);

	for(my $index=0; $index<@subs; $index++)
	{	my $subroutine = $subs[$index];
		my $show = $manual->inherited($subroutine) ? $args{show_inherited_subs} : $args{show_described_subs};

		$self->showSubroutine(
			%args,
			subroutine      => $subroutine,
			show_subroutine => $show,
			last            => ($index==$#subs),
		);
	}
}

=method showSubroutine %options
Show a single subroutine.

=requires subroutine OBJECT
=requires manual  $manual

=option  output  FILE
=default output  <selected filehandle>

=option  show_subroutine 'NO'|'NAMES'|'USE'|'EXPAND'
=default show_subroutine 'EXPAND'

=option  show_option_table 'NO'|'INHERITED'|'DESCRIBED'|'ALL'
=default show_option_table 'ALL'

=option  show_inherited_options 'NO'|'LIST'|'USE'|'EXPAND'
=default show_inherited_options 'USE'

=option  show_described_options 'NO'|'LIST'|'USE'|'EXPAND'
=default show_described_options 'EXPAND'

=option  show_sub_description 'NO'|'DESCRIBED'|'REFER'|'ALL'
=default show_sub_description 'DESCRIBED'
Included the description of the use of the subroutines, which
comes before the options are being explained.  P<NO> will cause
the description to be ignored, P<DESCRIBED> means that only
text which was written in the manual-page at hand is included,
P<REFER> means that a reference to inherited documentation is
made, and with P<ALL> the inherited texts are expanded into this
file as well.

=option  show_examples 'NO'|'EXPAND'
=default show_examples 'EXPAND'

=option  show_diagnostics 'NO'|'EXPAND'
=default show_diagnostics 'NO'
Diagnostics (error and warning messages) are defined per subroutine,
but are usually not listed with the subroutine.  The POD formatter's
default behavior, for instance, puts them all in a separate DIAGNOSTICS
chapter per manual page.

=option  last BOOLEAN
=default last false

=error illegal value for show_subroutine: $value
=error illegal value for show_sub_description: $value
=error illegal value for show_option_table: $value
=cut

sub showSubroutine(@)
{	my ($self, %args) = @_;

	my $subroutine = $args{subroutine} or panic;
	my $manual = $args{manual} or panic;
	my $output = $args{output} || select;

	#
	# Method use
	#

	my $use    = $args{show_subroutine} || 'EXPAND';
	my ($show_use, $expand)
	  = $use eq 'EXPAND' ? ('showSubroutineUse',  1)
	  : $use eq 'USE'    ? ('showSubroutineUse',  0)
	  : $use eq 'NAMES'  ? ('showSubroutineName', 0)
	  : $use eq 'NO'     ? (undef,                0)
	  :   error __x"illegal value for show_subroutine: {value}", value => $use;

	$self->$show_use(%args, subroutine => $subroutine)
		if defined $show_use;

	$expand or return;

	$args{show_inherited_options} ||= 'USE';
	$args{show_described_options} ||= 'EXPAND';

	#
	# Subroutine descriptions
	#

	my $descr       = $args{show_sub_description} || 'DESCRIBED';
	my $description = $subroutine->findDescriptionObject;
	my $show_descr  = 'showSubroutineDescription';

		if($descr eq 'NO') { $show_descr = undef }
	elsif($descr eq 'REFER')
	{	$show_descr = 'showSubroutineDescriptionRefer'
			if $description && $manual->inherited($description);
	}
	elsif($descr eq 'DESCRIBED')
	{	$show_descr = 'showSubroutineDescriptionRefer'
			if $description && $manual->inherited($description);
	}
	elsif($descr eq 'ALL') {;}
	else { error __x"illegal value for show_sub_description: {value}", value => $descr}

	$self->$show_descr(%args, subroutine => $description // $subroutine)
		if defined $show_descr;

	#
	# Options
	#

	my $options = $subroutine->collectedOptions;

	my $opttab  = $args{show_option_table} || 'NAMES';
	my @options = @{$options}{ sort keys %$options };

	# Option table

	my @opttab
	  = $opttab eq 'NO'       ? ()
	  : $opttab eq 'DESCRIBED'? (grep not $manual->inherits($_->[0]), @options)
	  : $opttab eq 'INHERITED'? (grep $manual->inherits($_->[0]), @options)
	  : $opttab eq 'ALL'      ? @options
	  :   error __x"illegal value for show_option_table: {value}", value => $opttab;

	$self->showOptionTable(%args, options => \@opttab) if @opttab;

	# Option expanded

	my @optlist;
	foreach (@options)
	{	my ($option, $default) = @$_;
		my $check = $manual->inherited($option) ? $args{show_inherited_options} : $args{show_described_options};
		push @optlist, $_ if $check eq 'USE' || $check eq 'EXPAND';
	}

	$self->showOptions(%args, options => \@optlist)
		if @optlist;

	# Examples

	my @examples = $subroutine->examples;
	my $show_ex  = $args{show_examples} || 'EXPAND';
	$self->showExamples(%args, examples => \@examples)
		if $show_ex eq 'EXPAND';

	# Diagnostics

	my @diags    = $subroutine->diagnostics;
	my $show_diag= $args{show_diagnostics} || 'NO';
	$self->showDiagnostics(%args, diagnostics => \@diags)
		if $show_diag eq 'EXPAND';
}

=method showExamples %options
=requires examples ARRAY
=requires manual MANUAL
=requires output FILE
=cut

sub showExamples(@) { $_[0] }

=method showSubroutineUse %options
=requires subroutine OBJECT
=requires manual OBJECT
=requires output FILE
=cut

sub showSubroutineUse(@) { $_[0] }

=method showSubroutineName %options
=requires subroutine OBJECT
=requires manual OBJECT
=requires output FILE

=option  last BOOLEAN
=default last 0
=cut

sub showSubroutineName(@) { $_[0] }

=method showSubroutineDescription %options
=requires subroutine OBJECT
=requires manual OBJECT
=requires output FILE
=cut

sub showSubroutineDescription(@) { $_[0] }

=method showOptionTable %options
=requires options ARRAY
=requires manual OBJECT
=requires output FILE
=cut

sub showOptionTable(@)
{	my ($self, %args) = @_;
	my $options = $args{options} or panic;
	my $manual  = $args{manual}  or panic;
	my $output  = $args{output}  or panic;

	my @rows;
	foreach (@$options)
	{	my ($option, $default) = @$_;
		my $optman = $option->manual;
		push @rows, [
			$self->cleanupString($manual, $option->name, tag => 'option_name'),
			($manual->inherited($option) ? $self->link(undef, $optman) : ''),
			$self->cleanupString($manual, $default->value, tag => 'option_default'),
		];
	}

	my @header  = ('Option', 'Defined in', 'Default');
	unless(grep length $_->[1], @rows)
	{	# removed empty "defined in" column
		splice @$_, 1, 1 for @rows, \@header;
	}

	$output->print("\n");
	$self->writeTable(output => $output, header => \@header, rows => \@rows, widths => [undef, 15, undef]);
	$self;
}

=method showOptions %options
The options shown are B<not> the %options passed as argument, but the
options which belong to the subroutine being displayed.

=requires options ARRAY
=requires manual OBJECT

=option  show_inherited_options 'NO'|'LIST'|'USE'|'EXPAND'
=default show_inherited_options 'USE'

=option  show_described_options 'NO'|'LIST'|'USE'|'EXPAND'
=default show_described_options 'EXPAND'

=error illegal show option choice: $value
=cut

sub showOptions(@)
{	my ($self, %args) = @_;

	my $options = $args{options} or panic;
	my $manual  = $args{manual}  or panic;

	foreach (@$options)
	{	my ($option, $default) = @$_;
		my $show = $manual->inherited($option) ? $args{show_inherited_options} : $args{show_described_options};

		my $action
		  = $show eq 'USE'   ? 'showOptionUse'
		  : $show eq 'EXPAND'? 'showOptionExpand'
		  :   error __x"illegal show option choice: {value}", value => $show;

		$self->$action(%args, option => $option, default => $default);
	}
	$self;
}

=method showOptionUse %options
=requires option OBJECT
=requires default OBJECT
=requires output FILE
=requires manual OBJECT
=cut

sub showOptionUse(@) { $_[0] }

=method showOptionExpand %options
=requires option OBJECT
=requires default OBJECT
=requires output FILE
=requires manual OBJECT
=cut

sub showOptionExpand(@) { $_[0] }

1;
