#!/usr/bin/perl -T
#
# Author : R.N. Bokhorst, November 2008
#
# This script is to be called from a mailserver like postfix.
#
# Its life-work is to collect XML jobs for the Forecast server that come in
#  through an email gateway. This enables our customers to send jobs in a simple email.
#
# This is a summary of what it does:
#   - Takes emails as input through STDIN
#   - Saves the message to disk
#   - Strips off the XML attachments and verifies their syntax
#   - Sends a confirmation email to the sender of the original email.
#
BEGIN{
    $ENV{PATH} = "/bin";
    if (not defined $ENV{HOSTNAME}) {
      my $hostname = `/bin/uname -n`;
      chomp $hostname;
      $ENV{HOSTNAME} = $hostname;
    }
}


use strict;
use warnings;
use English;

use POSIX qw(strftime);
use File::Copy;
use Mail::Message;
use Mail::Message::Attachment::Stripper;
use Mail::SendEasy;
use Email::Valid;
use Date::Parse qw(str2time);
use Data::Dumper;
use Fcntl; 

# Globals
my $mail = new Mail::SendEasy( smtp => 'mail.argoss.nl' );
my $now = time();

#-----------------------------------------------------------------------------#
# Config                                                                      #
#-----------------------------------------------------------------------------#
my $from_email = 'smart@jb-test.argoss.nl';
#my $helpdesk_email = 'operator@argoss.nl';
my $helpdesk_email = 'john.boers@bmtargoss.com,operator@argoss.nl';
my $from_title = 'smart@jb-test: error smart_mailgw.pl';

my $data_root_dir = "/home/markov/shared/perl/MailBox3/bmtargoss";
my $mail_dir = "${data_root_dir}/queue";
my $dbmt_dir = "${data_root_dir}/dbmt";

my $msg_fname = "${mail_dir}/message_${now}_" . $$ . "_in.txt";

#-- End config ---------------------------------------------------------------#


    
# Catch all errors and send to the helpdesk
TRY: eval {
  main();  
};
CATCH: if($@) {
  my $err = $EVAL_ERROR;
  chomp $err;
  eval { 
    $mail->send(
      from    => $from_email,
      to      => $helpdesk_email,
      subject => $from_title,
      message => qq(The following error occurred:\n\n) .
                 qq(  "$err"\n\n) .
                 qq(The original email message is saved as $msg_fname)
    );
  };

  # Enable for debugging
  die "\nERROR MESSAGE:\n" .
      "\n$err\nMessage was sent to $helpdesk_email\n\n";
}


#-----------------------------------------------------------------------------#
# MAIN                                                                        #
#-----------------------------------------------------------------------------#
sub main
{
  # Read the message from STDIN
  my $message;
  { local $/;
    $message = <STDIN>;
  }
  
  # Be carefull..
  #umask "0000";

  # Write message to file
  mkdir $mail_dir if not -d $mail_dir;
  sysopen(FH, "$msg_fname", O_WRONLY|O_CREAT|O_APPEND, 0644) or die "could not open $msg_fname: $!";
  #printf FH "\n\n%o\n\n", umask;
  print FH $message;
  close FH;

  # Transform to Mail::Message object
  my $mail_msg = Mail::Message->read( \$message );
  
exit 0;
  # Get sender address
  my $from = $mail_msg->sender()->address();
  die "Sender is not a valid email address" 
    if not Email::Valid->address(-address => $from, -mxcheck => 1);
  
  # Tear off the attachments 
  my $stripper = Mail::Message::Attachment::Stripper->new( $mail_msg );
  my $found = 0;
  my @attachments = $stripper->attachments;
  
  foreach my $att (@attachments) {

    # Attachment should be dbmt file with name like
    # LB<ship alias><date><time>.dbmt 

    my $att_fname = $att->{filename};
    $att_fname =~ tr/A-Z/a-z/;

    if ( $att_fname =~ /^lb.*\.dbmt$/ ) {

       $att_fname =~ s/\s/_/g;

       my $att_name = "${dbmt_dir}/$att_fname";

       # Dump attachment (usefull when base64 encoded in original message)
       sysopen(FH, "$att_name", O_WRONLY|O_CREAT|O_APPEND, 0644) or die "could not open $att_name: $!";
       print FH $att->{payload};
       close FH;

       # start processing here.... 

       $found = 1;
    }

  }

  die "Invalid SMART email; no valid DBMT attachment found." 
     if not $found;

}


