use warnings;
use strict;

package Mangle;
use base 'Exporter';

our @EXPORT = qw/mangle_header mangle_body strip_attachments/;

sub mangle_header($$$);
sub mangle_body($;$$);

sub _wrap_body_rules($;$);
sub _remove_attachment_rule($$@);
sub _substitute_values($$$);
sub _substitute_unsubscribe($$);

use Log::Report      qw(simplelists);
use feature 'state';
use Encode;
use HTML::Entities;

=pod

mangle_header MESSAGE, CONFIG, ACCOUNT

=cut

sub mangle_header($$$)
{   my ($msg, $list_config, $account) = @_;
    my $head = $msg->head;

    # Remove these headers
    $head->delete('x-confirm-reading-to');        # pegasus mail (windoze)
    $head->delete('disposition-notification-to'); # eudora
    $head->delete('x-ack');
    $head->delete('return-receipt-to');
    $head->delete('errors-to');
    $head->delete('flags');
    $head->delete('x-pmrqc');
    $head->delete('return-path');
    $head->delete('domainKey-signature');

    my ($listname, $domain) = @{$account}{qw(listname domain)};

    # Add subject prefix if not already added
    if(my $prefix = $list_config->{subject_prefix})
    {   my $foo = _substitute_values $prefix, $listname, $domain;

        # prepend subject prefix
        if(my $subject = $head->get('subject'))
        {   $head->set(Subject => "$foo $subject")
                 unless $subject =~ /\Q$foo/;
        }
        else # No existing subject line
        {   $head->set(Subject => $foo);
        }
    }

    # add new header fields
    $head->set(Precedence => "bulk");

    # Set new reply-to if need be
    if(my $rt = $list_config->{reply_to})
    {   $head->set('Reply-To' => _substitute_values $rt, $listname, $domain);
    }

    # Set new from if need be
    if(my $smf = $list_config->{set_message_from})
    {   $head->set(From => $smf);
    }

    # add per-list additonal headers
    if(my $mh = $list_config->{message_headers})
    {   my $headers = _substitute_values $mh, $listname, $domain;
	$head->set($_)
            for split /\n/, $headers;
    }

    $msg;
}

sub _wrap_body_rules($;$)
{   my $list_config = shift or return;
    my $unsubscribe_link = shift // '';
    
    my $mf = $list_config->{message_footer} // '';
    my $footer = length $mf ? (_substitute_unsubscribe $mf, $unsubscribe_link) : undef;

    my $footer_html;
    my $mfh = $list_config->{message_footer_html} // '';
    if(length $mfh)
    {   $footer_html = _substitute_unsubscribe $mfh, $unsubscribe_link;
    }
    elsif($footer)
    {   my $html = encode_entities $footer;
        $html    =~ s#\b(http://[[:alnum:][:punct:]]+)\b#<a href="$1">$1</a>#i;
        $footer_html = "\n<tt><pre>$html</pre></tt>";
    }

    my $fronter;
    my $mfr = $list_config->{message_fronter} // '';
    if(length $mfr)
    {   $fronter      = _substitute_unsubscribe $mfr, $unsubscribe_link;
    }

    my $fronter_html;
    my $mfrh = $list_config->{message_fronter_html} // '';
    if(length $mfrh)
    {   $fronter_html = _substitute_unsubscribe $mfrh, $unsubscribe_link;
    }
    elsif(defined $fronter)
    {   $fronter_html = encode_entities $fronter;
        $fronter_html =~ s#\b(http://[[:alnum:][:punct:]]+)\b#<a href="$1">$1</a>#i;
        $fronter_html =~ s/\n/<br>\n/g;
        $fronter_html = "<tt>$fronter_html</tt>";
    }

    my $need_wrap_text = defined $fronter || defined $footer;
    my $wrap_text = sub ($$@) {
       my ($msg, $part, %opts) = @_;
      
       $need_wrap_text && !$part->head->get('X-SL-Modified')
           or return $part;

       $part->body->mimeType eq 'text/plain'
           or return $part;

#      $need_wrap_text = 0;   # add this to mangle only first text

       # The mangled text should use the samen transfer-encoding, usually
       # quoted-printable when lines are too long.
       my $transenc = $part->body->transferEncoding;
       my $decoded  = $part->body->decoded;
       my $newpart  = (ref $part)->new
         ( head      => $part->head->clone
         , container => undef
         );
       # As should the footer/fronter
       my $charset = $part->body->charset || "latin1";
       $footer = encode($charset, $footer);
       $fronter = encode($charset, $fronter);

       my $newbody  = (ref $decoded)->new
         ( based_on  => $decoded
         , data      => ($fronter//'') . $decoded . ($footer//'')
	 , transfer  => 'none'
         )->encode(transfer_encoding => $transenc);
       $newpart->body($newbody);
       $newpart->head->add('X-SL-Modified' => 'wrapped_text');

       $newpart;
    };

    my $need_wrap_html = defined $fronter_html || defined $footer_html;
    my $wrap_html = sub ($$@) {
       my ($msg, $part, %opts) = @_;

       $need_wrap_html && !$part->head->get('X-SL-Modified')
           or return $part;

       $part->body->mimeType eq 'text/html'
           or return $part;

#      $need_wrap_html = 0;   # add this to mangle only first html
       my $decoded = $part->body->decoded;
       my $newpart = (ref $part)->new
         ( head      => $part->head->clone
         , container => undef
         );
       my $newbody = (ref $decoded)->new
         ( based_on  => $decoded
         , data      => ($fronter_html//'') . $decoded . ($footer_html//'')
         )->encode(transfer_encoding => 'quoted-printable');
       $newpart->body($newbody);
       $newpart->head->add('X-SL-Modified' => 'wrapped_html');
       $newpart;
    };

    ($wrap_text, $wrap_html);
}

sub _remove_attachment_rule($$@)
{   my ($msg, $part, %opts) = @_;

       $part->isMultipart
    || $part->isNested
    || ($part->body->isText && $part->body->mimeType ne 'text/calendar')
        ? $part : undef;
}

sub _remove_attachment_rule_nested($$@)
{   my ($msg, $part, %opts) = @_;
    my $body = $part->body;

    ($body->isText && !$part->isNested && $body->mimeType ne 'text/calendar')
      ? $part : undef;
}

sub mangle_body($;$$)
{   my ($msg, $list_config, $unsubscribe_link) = @_;

    state $rebuild;

    my $need_rebuild
       = !($rebuild && defined($list_config) && $unsubscribe_link)
      || $list_config->{message_footer} =~ /\$UNSUBSCRIBE/;
 
    # Prevent message being rebuilt every time if not needed
    if($need_rebuild)
    {   # See Mail::Message::Construct::Rebuild/Adding your own rules
        my @extra_rules = _wrap_body_rules $list_config, $unsubscribe_link;

        push @extra_rules, \&_remove_attachment_rule
            if $list_config->{strip_attachments};

        $rebuild = $msg->rebuild
          ( keep_message_id => 1
          , extra_rules     => \@extra_rules
          );
    }

    $rebuild;
}

sub strip_attachments($@)
{
    my ($msg, %opts) = @_;
    my $strip_nested = $opts{strip_nested} ? 1 : 0;
    my @extra_rules;
    if($strip_nested) { @extra_rules = \&_remove_attachment_rule_nested }
    else              { @extra_rules = \&_remove_attachment_rule        }
 
    $msg->rebuild
      ( keep_message_id => 1
      , extra_rules     => \@extra_rules
      );
}

sub _substitute_values($$$)
{    my ($string, $list, $domain) = @_;
     if(defined $string && $string =~ m/\$/)
     {   $string =~ s/(?<!\\)\$LIST/$list/g;
         $string =~ s/(?<!\\)\$DOMAIN/$domain/g;
     }
     $string;
}

sub _substitute_unsubscribe($$)
{    my ($string, $unsubscribe) = @_;
     if(defined $string && $string =~ m/\$/)
     {   $string =~ s/(?<!\\)\$UNSUBSCRIBE/$unsubscribe/g;
     }
     $string;
}

1;
