#oodist: *** DO NOT USE THIS VERSION FOR PRODUCTION ***
#oodist: This file contains OODoc-style documentation which will get stripped
#oodist: during its release in the distribution.  You can use this file for
#oodist: testing, however the code of this development version may be broken!

package Log::Report::Lexicon::PO;

use warnings;
use strict;

use Log::Report 'log-report-lexicon';

# steal from cheaper module, we have no ::Util for this (yet)
use Log::Report::Lexicon::POTcompact ();
*_escape   = \&Log::Report::Lexicon::POTcompact::_escape;
*_unescape = \&Log::Report::Lexicon::POTcompact::_unescape;

sub flat(@) { grep defined, ref $_[0] eq 'ARRAY' ? @{$_[0]} : @_ }

#--------------------
=chapter NAME
Log::Report::Lexicon::PO - one translation definition

=chapter SYNOPSIS

=chapter DESCRIPTION
This module is administering one translation object.  Sets of PO
records are kept in a POT file, implemented in Log::Report::Lexicon::POT.

=chapter METHODS

=section Constructors

=c_method new %options

=requires msgid STRING

=option   msgid_plural STRING
=default  msgid_plural undef

=option   msgstr STRING|ARRAY-OF-STRING
=default  msgstr "" or []
The translations for the msgid.  When msgid_plural is defined, then an
ARRAY must be provided.

=option   msgctxt STRING
=default  msgctxt undef
Context string: text around the msgid itself.

=option   comment PARAGRAPH
=default  comment []
Translator added comments.
See M<addComment()>.

=option   fuzzy BOOLEAN
=default  fuzzy false
The string is not yet translated, some smart guesses may have been made.
See M<fuzzy()>.

=option   automatic PARAGRAPH
=default  automatic ""
Automatically added comments.
See M<addAutomatic()>.

=option   references STRING|ARRAY-OF-LOCATIONS
=default  references []
The STRING is a blank separated list of LOCATIONS.
LOCATIONs are of the  form C<filename:linenumber>, for
instance C<lib/Foo.pm:42>
See M<addReferences()>

=option   format ARRAY|HASH
=default  format C<[]>
See M<format()>.  Either an ARRAY with PAIRS or a HASH with that same
information.

=error no msgid defined for PO
=cut

sub new(@)
{	my $class = shift;
	(bless {}, $class)->init( {@_} );
}

sub init($)
{	my ($self, $args) = @_;
	defined($self->{msgid} = delete $args->{msgid})
		or error __"no msgid defined for PO";

	$self->{plural}  = delete $args->{msgid_plural};
	$self->{msgstr}  = delete $args->{msgstr};
	$self->{msgctxt} = delete $args->{msgctxt};

	$self->addComment(delete $args->{comment});
	$self->addAutomatic(delete $args->{automatic});
	$self->fuzzy(delete $args->{fuzzy});

	$self->{refs}   = {};
	$self->addReferences(delete $args->{references})
		if defined $args->{references};

	$self;
}

# only for internal usage
sub _fast_new($) { bless $_[1], $_[0] }

#--------------------
=section Attributes

=method msgid
Returns the actual msgid, which cannot be undef.

=method msgctxt
Returns the message context, if provided.
=cut

sub msgid()   { $_[0]->{msgid} }
sub msgctxt() { $_[0]->{msgctxt} }

=method plural [STRING]
Returns the actual msgid_plural, which can be undef.
=cut

sub plural(;$)
{	my $self = shift;
	@_ or return $self->{plural};

	if(my $m = $self->{msgstr})
	{	# prepare msgstr list for multiple translations.
		$self->{msgstr} = [ $m ] if defined $m && !ref $m;
	}

	$self->{plural} = shift;
}

=method msgstr [$index, [STRING]]
With a STRING, a new translation will be set.  Without STRING, a
lookup will take place.  When no plural is defined, the $index is
ignored.
=cut

sub msgstr($;$)
{	my $self = shift;
	my $m    = $self->{msgstr};

	unless($self->{plural})
	{	$self->{msgstr} = $_[1] if @_==2;
		return $m;
	}

	my $index    = shift || 0;
	@_ ? $m->[$index] = shift : $m->[$index];
}

=method comment [LIST|ARRAY|STRING]
Returns a STRING which contains the cleaned paragraph of translator's
comment.  If an argument is specified, it will replace the current
comment.
=cut

sub comment(@)
{	my $self = shift;
	@_ or return $self->{comment};
	$self->{comment} = '';
	$self->addComment(@_);
}

=method addComment LIST|ARRAY|STRING
Add multiple lines to the translator's comment block.  Returns an
empty string if there are no comments.
=cut

sub addComment(@)
{	my $self    = shift;
	my @lines   = flat @_;
	my $comment = $self->{comment};

	foreach my $line (@lines)
	{	defined $line or next;
		$line =~ s/[\r\n]+/\n/;  # cleanup line-endings
		$comment .= $line;
	}

	# be sure there is a \n at the end
	$comment =~ s/\n?\z/\n/ if defined $comment;
	$self->{comment} = $comment;
}

=method automatic [LIST|ARRAY|STRING]
Returns a STRING which contains the cleaned paragraph of automatically
added comments.  If an argument is specified, it will replace the current
comment.
=cut

sub automatic(@)
{	my $self = shift;
	@_ or return $self->{automatic};
	$self->{automatic} = '';
	$self->addAutomatic(@_);
}

=method addAutomatic LIST|ARRAY|STRING
Add multiple lines to the translator's comment block.  Returns an
empty string if there are no comments.
=cut

sub addAutomatic(@)
{	my $self  = shift;
	my @lines = flat @_;

	my $auto = $self->{automatic};
	foreach my $line (@lines)
	{	$line =~ s/[\r\n]+/\n/;  # cleanup line-endings
		$auto .= $line;
	}

	$auto =~ s/\n?\z/\n/ if defined $auto; # be sure there is a \n at the end
	$self->{automatic} = $auto;
}

=method references [STRING|LIST|ARRAY]
Returns an unsorted list of LOCATIONS.  When options are specified,
then those will be used to replace all currently defined references.
Returns the unsorted LIST of references.
=cut

sub references(@)
{	my $self = shift;
	if(@_)
	{	$self->{refs} = +{ };
		$self->addReferences(@_);
	}

	keys %{$self->{refs}};
}

=method addReferences STRING|LIST|ARRAY
The STRING is a blank separated list of LOCATIONS.  The LIST and
ARRAY contain separate LOCATIONs.  A LOCATION is of the form
C<filename:linenumber>.  Returns the internal HASH with references.
=cut

sub addReferences(@)
{	my $self = shift;
	my $refs = $self->{refs} ||= {};

	my @new = @_==1 && defined $_[0] && ref $_[0] ne 'ARRAY' ? split(" ", $_[0]) : flat @_;
	$refs->{$_}++ for @new;
	$refs;
}

=method removeReferencesTo $filename
Remove all the references to the indicate $filename from the list.  Returns
the number of refs left.
=cut

sub removeReferencesTo($)
{	my $refs  = $_[0]->{refs};
	my $match = qr/^\Q$_[1]\E\:[0-9]+$/;
	$_ =~ $match && delete $refs->{$_}
		for keys %$refs;

	scalar keys %$refs;
}

=method keepReferencesTo $table
Remove all references which are not found as key in the hash $table.
Returns the number of references left.
=cut

sub keepReferencesTo($)
{	my $refs  = shift->{refs};
	my $keep  = shift;

	foreach my $ref (keys %$refs)
	{	(my $fn = $ref) =~ s/\:[0-9]+$//;
		$keep->{$fn} or delete $refs->{$ref};
	}

	scalar keys %$refs;
}

=method isActive
Returns whether the translation has any references, or is the header.
=cut

sub isActive() { $_[0]->{msgid} eq '' || keys %{$_[0]->{refs}} }

=method fuzzy [BOOLEAN]
Returns whether the translation needs human inspection.
=cut

sub fuzzy(;$) { my $self = shift; @_ ? $self->{fuzzy} = shift : $self->{fuzzy} }

=method format $language | <PAIRS|ARRAY|HASH>
When one $language is specified, it looks whether a "language-format" or
"no-language-format" is present in the line of FLAGS.  This will return
C<1> (true) in the first case, C<0> (false) in the second case.  It will
return undef (also false) in case that both are not present.

You can also specify PAIRS: the key is a language name, and the
value is either C<0>, C<1>, or undef.

=examples use of format()
  # getters
  if($po->format('c')) ...
  unless($po->format('perl-brace')) ...
  if(defined $po->format('java')) ...

  # setters
  $po->format(java => 1);       # results in 'java-format'
  $po->format(java => 0);       # results in 'no-java-format'
  $po->format(java => undef);   # results in ''
=cut

sub format(@)
{	my $format = shift->{format};
	return $format->{ (shift) }
		if @_==1 && !ref $_[0];  # language

	my @pairs = ref $_[0] eq 'HASH' ? %{$_[0]} : flat @_;
	while(@pairs)
	{	my($k, $v) = (shift @pairs, shift @pairs);
		$format->{$k} = $v;
	}
	$format;
}

=method addFlags STRING
Parse a "flags" line.
=cut

=warning unknown flag $flag ignored
=cut

sub addFlags($)
{	my $self  = shift;
	local $_  = shift;
	my $where = shift;

	s/^\s+//;
	s/\s*$//;
	foreach my $flag (split /\s*\,\s*/)
	{	   if($flag eq 'fuzzy') { $self->fuzzy(1) }
		elsif($flag =~ m/^no-(.*)-format$/) { $self->format($1, 0) }
		elsif($flag =~ m/^(.*)-format$/)    { $self->format($1, 1) }
		else
		{	warning __x"unknown flag {flag} ignored", flag => $flag;
		}
	}
	$_;
}

#--------------------
=section Parsing

=c_method fromText STRING, [$where]
Parse the STRING into a new PO object.  The $where string should explain
the location of the STRING, to be used in error messages.

=warning unknown comment type '$cmd' at $where
=warning do not understand command '$cmd' at $where
=warning quoted line is not a continuation at $where
=warning do not understand line at $where:\n  $line
=warning no msgid in block $where
=cut

sub fromText($$)
{	my $class = shift;
	my @lines = split /[\r\n]+/, shift;
	my $where = shift || ' unknown location';

	my $self  = bless +{}, $class;

	# translations which are not used anymore are escaped with #~
	# however, we just say: no references found.
	s/^\#\~\s+// for @lines;

	my $last;  # used for line continuations
	foreach (@lines)
	{	s/\r?\n$//;
		if( s/^\#(.)\s?// )
		{	   if($1 =~ /\s/) { $self->addComment($_)    }
			elsif($1 eq '.' ) { $self->addAutomatic($_)  }
			elsif($1 eq ':' ) { $self->addReferences($_) }
			elsif($1 eq ',' ) { $self->addFlags($_)      }
			else
			{	warning __x"unknown comment type '{cmd}' at {where}", cmd => "#$1", where => $where;
			}
			undef $last;
		}
		elsif( s/^\s*(\w+)\s+// )
		{	my $cmd    = $1;
			my $string = _unescape($_,$where);

			if($cmd eq 'msgid')
			{	$self->{msgid} = $string;
				$last = \($self->{msgid});
			}
			elsif($cmd eq 'msgid_plural')
			{	$self->{plural} = $string;
				$last = \($self->{plural});
			}
			elsif($cmd eq 'msgstr')
			{	$self->{msgstr} = $string;
				$last = \($self->{msgstr});
			}
			elsif($cmd eq 'msgctxt')
			{	$self->{msgctxt} = $string;
				$last = \($self->{msgctxt});
			}
			else
			{	warning __x"do not understand command '{cmd}' at {where}", cmd => $cmd, where => $where;
				undef $last;
			}
		}
		elsif( s/^\s*msgstr\[(\d+)\]\s*// )
		{	my $nr = $1;
			$self->{msgstr}[$nr] = _unescape($_,$where);
		}
		elsif( m/^\s*\"/ )
		{	if(defined $last) { $$last .= _unescape($_, $where) }
			else
			{	warning __x"quoted line is not a continuation at {where}", where => $where;
			}
		}
		else
		{	warning __x"do not understand line at {where}:\n  {line}", where => $where, line => $_;
		}
	}

	defined $self->{msgid}
		or warning __x"no msgid in block {where}", where => $where;

	$self;
}

=method toString %options
Format the object into a multi-lined string.

=option  nr_plurals INTEGER
=default nr_plurals undef
If the number of plurals is specified, then the plural translation
list can be checked for the correct length.  Otherwise, no smart
behavior is attempted.

=warning too many plurals for '$msgid'
=warning no plurals for '$msgid'
=cut

sub toString(@)
{	my ($self, %args) = @_;
	my $nplurals = $args{nr_plurals};
	my @record;

	my $comment = $self->comment;
	if(defined $comment && length $comment)
	{	$comment =~ s/^/#  /gm;
		push @record, $comment;
	}

	my $auto = $self->automatic;
	if(defined $auto && length $auto)
	{	$auto =~ s/^/#. /gm;
		push @record, $auto;
	}

	my @refs    = sort $self->references;
	my $msgid   = $self->{msgid} || '';
	my $active  = $msgid eq ''   || @refs ? '' : '#~ ';

	while(@refs)
	{	my $line = '#:';
		$line .= ' '.shift @refs while @refs && length($line) + length($refs[0]) < 80;
		push @record, "$line\n";
	}

	my @flags   = $self->{fuzzy} ? 'fuzzy' : ();

	push @flags, ($self->{format}{$_} ? '' : 'no-') . $_ . '-format'
		for sort keys  %{$self->{format}};

	push @record, "#, ". join(", ", @flags) . "\n"
		if @flags;

	my $msgctxt = $self->{msgctxt};
	if(defined $msgctxt && length $msgctxt)
	{	push @record, "${active}msgctxt "._escape($msgctxt, "\n$active")."\n";
	}
	push @record, "${active}msgid "._escape($msgid, "\n$active")."\n";

	my $msgstr  = $self->{msgstr} || [];
	my @msgstr  = ref $msgstr ? @$msgstr : $msgstr;
	my $plural  = $self->{plural};
	if(defined $plural)
	{	push @record, "${active}msgid_plural " . _escape($plural, "\n$active") . "\n";

		push @msgstr, ''
			while defined $nplurals && @msgstr < $nplurals;

		if(defined $nplurals && @msgstr > $nplurals)
		{	warning __x"too many plurals for '{msgid}'", msgid => $msgid;
			$#msgstr = $nplurals -1;
		}

		$nplurals ||= 2;
		for(my $nr = 0; $nr < $nplurals; $nr++)
		{	push @record, "${active}msgstr[$nr] " . _escape($msgstr[$nr], "\n$active") . "\n";
		}
	}
	else
	{	warning __x"no plurals for '{msgid}'", msgid => $msgid if @msgstr > 1;
		push @record, "${active}msgstr " . _escape($msgstr[0], "\n$active") . "\n";
	}

	join '', @record;
}

=method useless
[1.02] The message-id has no references anymore B<and> no translations.
=cut

sub useless()
{	my $self = shift;
	! $self->references && ! $self->msgstr(0);
}
*unused = \&useless; # before <1.02

1;
